
/*******************************************************************************/
/*  
	This script produces Figure 6
	Mongey, Pilossoph, and Weinberg: "Which Workers Bear the Burden of Social Distancing?"
	Journal of Economic Inequality 2021	
    --
	Alexander Weinberg
	February 16, 2021
*/
/*******************************************************************************/

// __________________________________________________
// Initialize empty dataset for later
clear all 
set obs 1
gen group = "."
save "../Data/CPS/dlogE_by_group", replace


// __________________________________________________
// Adjust for inflation
import delimited "../Raw Data/CPIAUCSL.csv", clear 	// Download from FRED
gen year = substr(date, 1, 4)
destring year, replace

sort year 
replace cpi = cpi[_n-1] if mi(cpi) & year == 2020 	// note 2020 CPI unavailable

qui summ cpi if (year==2019)						// normalize to 2019 prices
local cpi2019 = r(mean)
replace cpi = cpi/`cpi2019'
keep year cpi 

// __________________________________________________
// CPS data
merge 1:m year using "../Data/CPS/monthly_cps_2010_2020.dta"	// Merge in basic monthly CPS 2010-2020
drop if _merge == 1
keep if labforce == 2 // in labor force

// __________________________________________________
// Generate Variables
gen everyone 		= 1
gen nonmarried 		= marst != 1							// Married = spouse present

gen parttime 		= (uhrswork1 < 30) 						// Fulltime = 30 hr/week
replace parttime 	= . if (uhrswork1 == 0) | (uhrswork1 >= 997) 

gen female 			= (sex == 2)
gen nonwhite		= (race != 100)
gen young 			= (age < 50)

gen nocollege 		= !inrange(educ, 111, 125)
replace nocollege 	= . if educ > 125

gen bornabroad 		= (bpl != 9900)
gen noncitizen      = (citizen == 5)

replace earnweek  	= . if earnweek >= 9999
replace earnweek 	= earnweek / cpi

replace uhrswork1 	= . if uhrswork1 > 99
replace uhrsworkorg = . if uhrsworkorg > 99

// __________________________________________________
// Wages
gen fedminwage 		= 7.25
gen wage 			= earnweek / uhrswork1 
replace wage 		= . if wage < 0.5 * fedminwage 			// very low earners
replace wage 		= . if (earnweek>0) & (uhrswork1==0)

xtile inc_tile		= wage if !mi(wage), nq(2) 
gen bottom50		= (inc_tile  == 1) if !mi(inc_tile)

// __________________________________________________
/* For each worker group: compute dlogE feb -> april/august. */
foreach group of varlist everyone parttime nonmarried female nonwhite young bottom50 nocollege bornabroad noncitizen {
	preserve

		// -----------------------------------------
		// Employment by {month} x {group}
		drop if mi(`group')
		collapse (sum) employed [pw=compwt], by(month year `group') fast
			
		// -----------------------------------------
		// LOG EMP variables
		sort `group' year month 
		gen E_feb = employed if month==2
		gen E_apr = employed if month==4
		gen E_aug = employed if month==8

		by `group' year: replace E_feb = E_feb[_n-1] if mi(E_feb)
		by `group' year: replace E_apr = E_apr[_n-1] if mi(E_apr)

		// -----------------------------------------
		// Dlog Employment
		bys year: gen dlogE_apr 		= log(E_apr) - log(E_feb) if month == 4
		bys year: gen dlogE_aug 		= log(E_aug) - log(E_feb) if month == 8
		
		sort `group' year month 
		by `group' year: replace dlogE_apr = dlogE_apr[_n-1] if mi(dlogE_apr)
		keep if month==8

		// -----------------------------------------
		// Adjust for decade trend
		bys `group': egen mean_dlogE_apr_10_19 	 = mean(dlogE_apr) 	if (year < 2020)
		bys `group': egen mean_dlogE_aug_10_19 	 = mean(dlogE_aug) 	if (year < 2020)

		bys	`group': replace mean_dlogE_apr_10_19 = mean_dlogE_apr_10_19[_n-1] if mi(mean_dlogE_apr_10_19) 
		bys	`group': replace mean_dlogE_aug_10_19 = mean_dlogE_aug_10_19[_n-1] if mi(mean_dlogE_aug_10_19) 

		// Adj 2020 numbers by decade average
		gen dlogE_apr_adj     							 = dlogE_apr - mean_dlogE_apr_10_19
		gen dlogE_aug_adj     							 = dlogE_aug - mean_dlogE_aug_10_19

		keep if year==2020
		keep dlogE_*_adj `group'

		gen group = "`group'"
		qui reshape wide dlogE_*_adj, i(group) j(`group')

		append using "../Data/CPS/dlogE_by_group", force
		save "../Data/CPS/dlogE_by_group", replace
	restore
}

/*----------------------------------------------------*/
   /* [>    Plots   <] */ 
/*----------------------------------------------------*/

graph drop _all

// __________________________________________________
// Load data
use "../Data/CPS/dlogE_by_group", clear
drop if mi(dlogE_apr_adj1)

// __________________________________________________
// Relabel
replace group = "Non US citizen" if group == "noncitizen"
replace group = "Born outside US" if group == "bornabroad"
replace group = "No college degree" if group == "nocollege"
replace group = "Top 1% Income"  if group == "top1"
replace group = "Below median wage*" if group == "bottom50"
replace group = "Age < 50" if group == "young"
replace group = "Non-white" if group == "nonwhite"
replace group = "Female" if group == "female"
replace group = "Single" if group == "nonmarried"
replace group = "Part time employed*" if group == "parttime"

list dlogE_*_adj1 if group == "everyone" // Smaller than BLS number, but we drop many more ppl.

// __________________________________________________
// Sort by Feb-Apr change in log emp. 
sort dlogE_apr_adj1
gen my_order 			= _n 		// reorder
replace my_order 		= 12  if group  == "Part time employed*" 		// Different sample
replace my_order 		= 13  if group == "Below median wage*"			// Only outgoing rotation group

replace my_order 		= 99  if group == "everyone"					// Create a blank for spacing
replace dlogE_apr_adj1 	= .   if group == "everyone"
replace dlogE_aug_adj1 	= .   if group == "everyone"
replace group 			= " " if group == "everyone"
sort my_order

// __________________________________________________
// Compare to BLS p.6, https://www.bls.gov/news.release/archives/empsit_05082020.pdf
gen BLS_E_a2020 = 133403000
gen BLS_E_f2020 = 158759000
gen dlogE_BLS = log(BLS_E_a2020) - log(BLS_E_f2020)
qui summ dlogE_BLS
display as text "DlogE Feb-Apr (BLS) = " as result `r(mean)'

// __________________________________________________
// PLOT DLOGE APRIL
format dlogE_*_adj? %3.2f
graph dot dlogE_apr_adj0 dlogE_apr_adj1 ///
	, ///
	legend(order(2 "Yes" 1 "No") size(medium) cols(2) pos(6)) ///
	yline( -.1471267, lc(black) lp(dash)) ///
	yscale(range(-0.3 0.0)) ylabel(-0.3(0.1)0.00) ///
	ytitle("Feb-Apr log change in employment", size(large)) ///
	marker(1, mcolor(red) ms(X) msize(huge)) ///
	marker(2, mcolor(green) ms(O) msize(large)) ///
	over(group, sort(my_order) label(labsize(large))) exclude0 ///
	xsize(5) ysize(4) ///
	name(A1) ///
	ylabel(,labsize(large) format(%8.2f))  ///
	title("A. Feb-Apr Employment")


// __________________________________________________
// PLOT DLOGE AUGUST
format dlogE_*_adj? %3.2f
graph ///
	dot dlogE_aug_adj0 dlogE_aug_adj1 ///
	, ///
	legend(order(2 "Yes" 1 "No") size(medium) cols(2) pos(6)) ///
	yline(-.0709985, lc(black) lp(dash)) ///
	yscale(range(-0.3 0.0)) ylabel(-0.3(0.1)0.00) ///
	ytitle("Feb-Aug log change in employment", size(large)) ///
	marker(1, mcolor(red) ms(X) msize(huge)) ///
	marker(2, mcolor(green) ms(O) msize(large)) ///
	over(group, sort(my_order) label(labsize(large))) exclude0 ///
	xsize(5) ysize(4) ///
	name(A2) ///
	ylabel(,labsize(large) format(%8.2f))  ///
	title("B. Feb-Aug Employment")
	
graph combine A1 A2, xcommon ycommon
graph export "../Figures/fig6.eps", replace





//end
